/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.program.model.data;

import ghidra.docking.settings.Settings;
import ghidra.program.model.mem.MemBuffer;

/**
 *  Special dataType used only for function return types.  Used to indicate that
 * a function has no return value.
 */
public class VoidDataType extends BuiltIn {

	/** A statically defined DefaultDataType used when an Undefined byte is needed.*/
	public static VoidDataType dataType = new VoidDataType();

	public VoidDataType() {
		this(null);
	}

	public VoidDataType(DataTypeManager dtm) {
		super(null, "void", dtm);
	}

	/**
	 * 
	 * @see ghidra.program.model.data.DataType#getMnemonic(Settings)
	 */
	@Override
	public String getMnemonic(Settings settings) {
		return "void";
	}

	/**
	 * 
	 * @see ghidra.program.model.data.DataType#getLength()
	 */
	@Override
	public int getLength() {
		return 0;
	}

	/**
	 * 
	 * @see ghidra.program.model.data.DataType#getDescription()
	 */
	@Override
	public String getDescription() {
		return "void datatype";
	}

	/**
	 * 
	 * @see ghidra.program.model.data.DataType#getRepresentation(MemBuffer, Settings, int)
	 */
	@Override
	public String getRepresentation(MemBuffer buf, Settings settings, int length) {
		return "";
	}

	/**
	 * 
	 * @see ghidra.program.model.data.DataType#getValue(ghidra.program.model.mem.MemBuffer, ghidra.docking.settings.Settings, int)
	 */
	@Override
	public Object getValue(MemBuffer buf, Settings settings, int length) {
		return null;
	}

	@Override
	public DataType clone(DataTypeManager dtm) {
		if (dtm == getDataTypeManager()) {
			return this;
		}
		return new VoidDataType(dtm);
	}

	@Override
	public String getCTypeDeclaration(DataOrganization dataOrganization) {
		return null; // standard C name and type
	}

	/**
	 * Determine if the specified {@link DataType} is a {@link VoidDataType} after 
	 * stripping away any {@link TypeDef}.
	 * @param dt datatype to be tested
	 * @return true if dt is a void type
	 */
	public static boolean isVoidDataType(DataType dt) {
		if (dt == null) {
			return false;
		}
		if (dt instanceof TypeDef t) {
			dt = t.getBaseDataType();
		}
		return dt instanceof VoidDataType;
	}

}
