/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.util;

import ghidra.framework.store.LockException;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOutOfBoundsException;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressRangeIterator;
import ghidra.program.model.listing.Program;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockException;
import ghidra.program.util.MemoryBlockDiff;
import ghidra.program.util.ProgramConflictException;
import ghidra.program.util.ProgramMemoryComparator;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.exception.NotFoundException;
import ghidra.util.task.TaskMonitor;
import java.util.ArrayList;

public class MemoryDiff {
    private Program program1;
    private Program program2;
    private Memory memory1;
    private Memory memory2;
    private AddressRange[] ranges;
    private MemoryBlockDiff[] diffs;

    public MemoryDiff(Program p1, Program p2) throws ProgramConflictException {
        this.program1 = p1;
        this.program2 = p2;
        this.memory1 = this.program1.getMemory();
        this.memory2 = this.program2.getMemory();
        this.computeRanges();
        this.computeDifferences();
    }

    public Program getProgram1() {
        return this.program1;
    }

    public Program getProgram2() {
        return this.program2;
    }

    private void computeRanges() throws ProgramConflictException {
        ProgramMemoryComparator memComp = new ProgramMemoryComparator(this.program1, this.program2);
        ArrayList<AddressRange> rangeList = new ArrayList<AddressRange>();
        AddressRangeIterator rangeIter = memComp.getAddressRanges();
        while (rangeIter.hasNext()) {
            rangeList.add((AddressRange)rangeIter.next());
        }
        this.ranges = rangeList.toArray(new AddressRange[rangeList.size()]);
    }

    public int getNumRanges() {
        return this.ranges.length;
    }

    public AddressRange getRange(int index) {
        return this.ranges[index];
    }

    public MemoryBlockDiff getDifferenceInfo(int index) {
        return this.diffs[index];
    }

    private void computeDifferences() {
        this.diffs = new MemoryBlockDiff[this.ranges.length];
        for (int i = 0; i < this.ranges.length; ++i) {
            Address addr = this.ranges[i].getMinAddress();
            MemoryBlock block1 = this.memory1.getBlock(addr);
            MemoryBlock block2 = this.memory2.getBlock(addr);
            this.diffs[i] = new MemoryBlockDiff(block1, block2);
        }
    }

    public String getDifferences(Address p1Address) {
        int index = this.getAddressRangeIndex(p1Address);
        if (index < 0 || index >= this.diffs.length) {
            return null;
        }
        MemoryBlockDiff info = this.getDifferenceInfo(index);
        return info.getDifferencesAsString();
    }

    private int getAddressRangeIndex(Address address) {
        int low = 0;
        int high = this.diffs.length - 1;
        while (low <= high) {
            int mid = low + high >> 1;
            AddressRange range = this.ranges[mid];
            if (range.contains(address)) {
                return mid;
            }
            if (address.compareTo((Object)range.getMinAddress()) < 0) {
                high = mid - 1;
                continue;
            }
            low = mid + 1;
        }
        return -(low + 1);
    }

    public AddressRange[] getDifferentAddressRanges() {
        ArrayList<AddressRange> rangeDiffs = new ArrayList<AddressRange>();
        for (AddressRange range : this.ranges) {
            MemoryBlock block2;
            Address addr = range.getMinAddress();
            MemoryBlock block1 = this.memory1.getBlock(addr);
            if (this.sameMemoryBlock(block1, block2 = this.memory2.getBlock(addr))) continue;
            rangeDiffs.add(range);
        }
        return rangeDiffs.toArray(new AddressRange[rangeDiffs.size()]);
    }

    private boolean sameMemoryBlock(MemoryBlock block1, MemoryBlock block2) {
        if (block1 == null) {
            return block2 == null;
        }
        if (block2 == null) {
            return false;
        }
        if (!block1.getName().equals(block2.getName())) {
            return false;
        }
        if (!block1.getStart().equals((Object)block2.getStart())) {
            return false;
        }
        if (!block1.getEnd().equals((Object)block2.getEnd())) {
            return false;
        }
        if (block1.getSize() != block2.getSize()) {
            return false;
        }
        if (block1.getFlags() != block2.getFlags()) {
            return false;
        }
        if (!block1.getType().equals((Object)block2.getType())) {
            return false;
        }
        if (block1.isInitialized() != block2.isInitialized()) {
            return false;
        }
        if (!SystemUtilities.isEqual((Object)block1.getSourceName(), (Object)block2.getSourceName())) {
            return false;
        }
        if (!SystemUtilities.isEqual((Object)block1.getComment(), (Object)block2.getComment())) {
            return false;
        }
        return block1.isMapped() == block2.isMapped();
    }

    public boolean merge(int row, int mergeFields, TaskMonitor monitor) {
        if ((mergeFields & 0x1FFF) == 0) {
            return false;
        }
        if (row < 0 || row >= this.diffs.length) {
            return false;
        }
        MemoryBlockDiff blockDiff = this.diffs[row];
        MemoryBlock block1 = blockDiff.getBlock1();
        MemoryBlock block2 = blockDiff.getBlock2();
        AddressRange range = this.ranges[row];
        if (this.shouldMerge(mergeFields, 2) && blockDiff.isStartAddressDifferent()) {
            if (block1 == null) {
                Address start2 = block2.getStart();
                Address end2 = block2.getEnd();
                Address startRange = range.getMinAddress();
                Address endRange = range.getMaxAddress();
                int compareStart = start2.compareTo((Object)startRange);
                int compareEnd = end2.compareTo((Object)endRange);
                try {
                    MemoryBlock secondBlock;
                    MemoryBlock firstBlock;
                    this.memory1.createBlock(block2, block2.getName(), startRange, range.getLength());
                    if (compareStart < 0) {
                        firstBlock = this.memory1.getBlock(start2);
                        secondBlock = this.memory1.getBlock(startRange);
                        this.memory1.join(firstBlock, secondBlock);
                    }
                    if (compareEnd > 0) {
                        firstBlock = this.memory1.getBlock(endRange);
                        secondBlock = this.memory1.getBlock(end2);
                        this.memory1.join(firstBlock, secondBlock);
                    }
                    return true;
                }
                catch (Exception e) {
                    Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
                    return false;
                }
            }
            if (block2 == null) {
                Address start1 = block1.getStart();
                Address end1 = block1.getEnd();
                Address startRange = range.getMinAddress();
                Address endRange = range.getMaxAddress();
                int compareStart = start1.compareTo((Object)startRange);
                int compareEnd = end1.compareTo((Object)endRange);
                try {
                    if (compareEnd > 0) {
                        this.memory1.split(block1, endRange.add(1L));
                    }
                    if (compareStart < 0) {
                        this.memory1.split(block1, startRange);
                    }
                    if (compareStart == 0 && compareEnd == 0) {
                        MemoryBlock blockToRemove = this.memory1.getBlock(startRange);
                        this.memory1.removeBlock(blockToRemove, monitor);
                    }
                    return true;
                }
                catch (LockException e) {
                    Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
                }
                catch (NotFoundException e) {
                    Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
                }
                catch (AddressOutOfBoundsException e) {
                    Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
                }
                catch (MemoryBlockException e) {
                    Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
                }
                return false;
            }
        }
        if (!this.shouldMerge(mergeFields, 4) || blockDiff.isEndAddressDifferent()) {
            // empty if block
        }
        if (!this.shouldMerge(mergeFields, 8) || blockDiff.isSizeDifferent()) {
            // empty if block
        }
        if (!this.shouldMerge(mergeFields, 512) || blockDiff.isTypeDifferent()) {
            // empty if block
        }
        if (!this.shouldMerge(mergeFields, 1024) || blockDiff.isInitDifferent()) {
            // empty if block
        }
        if (this.shouldMerge(mergeFields, 1) && blockDiff.isNameDifferent()) {
            try {
                block1.setName(block2.getName());
            }
            catch (LockException e) {
                Msg.error((Object)this, (Object)("Unexpected Exception: " + e.getMessage()), (Throwable)e);
            }
        }
        if (this.shouldMerge(mergeFields, 16) && blockDiff.isReadDifferent()) {
            block1.setRead(block2.isRead());
        }
        if (this.shouldMerge(mergeFields, 32) && blockDiff.isWriteDifferent()) {
            block1.setWrite(block2.isWrite());
        }
        if (this.shouldMerge(mergeFields, 64) && blockDiff.isExecDifferent()) {
            block1.setExecute(block2.isExecute());
        }
        if (this.shouldMerge(mergeFields, 128) && blockDiff.isVolatileDifferent()) {
            block1.setVolatile(block2.isVolatile());
        }
        if (this.shouldMerge(mergeFields, 256) && blockDiff.isArtificialDifferent()) {
            block1.setArtificial(block2.isArtificial());
        }
        if (this.shouldMerge(mergeFields, 2048) && blockDiff.isSourceDifferent()) {
            block1.setSourceName(block2.getSourceName());
        }
        if (this.shouldMerge(mergeFields, 4096) && blockDiff.isCommentDifferent()) {
            block1.setComment(block2.getComment());
        }
        return true;
    }

    private boolean shouldMerge(int mergeFields, int memDiffType) {
        return (mergeFields & memDiffType) != 0;
    }
}

