﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/DocumentFormat.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class GetDocumentRequest : public SSMRequest {
 public:
  AWS_SSM_API GetDocumentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetDocument"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the SSM document.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  GetDocumentRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional field specifying the version of the artifact associated with the
   * document. For example, 12.6. This value is unique across all versions of a
   * document and can't be changed.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  GetDocumentRequest& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document version for which you want information.</p>
   */
  inline const Aws::String& GetDocumentVersion() const { return m_documentVersion; }
  inline bool DocumentVersionHasBeenSet() const { return m_documentVersionHasBeenSet; }
  template <typename DocumentVersionT = Aws::String>
  void SetDocumentVersion(DocumentVersionT&& value) {
    m_documentVersionHasBeenSet = true;
    m_documentVersion = std::forward<DocumentVersionT>(value);
  }
  template <typename DocumentVersionT = Aws::String>
  GetDocumentRequest& WithDocumentVersion(DocumentVersionT&& value) {
    SetDocumentVersion(std::forward<DocumentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Returns the document in the specified format. The document format can be
   * either JSON or YAML. JSON is the default format.</p>
   */
  inline DocumentFormat GetDocumentFormat() const { return m_documentFormat; }
  inline bool DocumentFormatHasBeenSet() const { return m_documentFormatHasBeenSet; }
  inline void SetDocumentFormat(DocumentFormat value) {
    m_documentFormatHasBeenSet = true;
    m_documentFormat = value;
  }
  inline GetDocumentRequest& WithDocumentFormat(DocumentFormat value) {
    SetDocumentFormat(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_versionName;

  Aws::String m_documentVersion;

  DocumentFormat m_documentFormat{DocumentFormat::NOT_SET};
  bool m_nameHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_documentVersionHasBeenSet = false;
  bool m_documentFormatHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
